# SoupAutomat - implementation by Mark Jeronimus
# Version 1.0
# Version 1.1 by Frank Buss: fixed algorithm and initial line enter
#
# Une idee d'automate cellulaire a une seule dimension
# (un lien menant ici a ete publie sur SeqFans le 13 janvier 2010)
#
# See: http://www.cetteadressecomportecinquantesignes.com/AutomateNBR01.htm
#
# Supported keys:
#  " "   - next gen
#  TAB   - next step
#  Enter - start/stop stepping
#  z     - undo
#  ESC   - abort script

import golly as g
from glife import rect
import time

undoHistory = list()

def update(iterations):
	if g.getoption("autofit") == 1:
		g.fit()
	else:
		g.update()

	if iterations > 0:
		undoHistory.append(iterations)

	g.show(str(len(undoHistory)))

# calculate next line, using the rules of the SoupAutomat
# line: Array of integers, with -1 for dot
# result: Array of integers of the same length as the input line
def calculateNextLine(line):
	newLine = [-1] * len(line)
	length = len(line)
	for i in xrange(length):
		c = line[i]
		if c >= 0:
			# calculate update postion
			if c % 2 == 0:
				# even
				ofs = c
			else:
				# odd
				ofs = -c
			
			# update cell
			n = i + ofs
			c = c + 1
			if newLine[n] == -1:
				newLine[n] = c
			else:
				newLine[n] = newLine[n] + c
				
	# fix overflows
	carry = -1
	for i in xrange(length):
		cell = newLine[i]
		if cell >= 0:
			if cell > 9:
				(cell, nextCarry) = (cell / 10, cell % 10)
			else:
				nextCarry = -1
			if carry >= 0:
				cell = cell + carry
				if cell > 9:
					(cell, rest) = (cell / 10, cell % 10)
					if nextCarry >= 0:
						nextCarry = nextCarry + rest
					else:
						nextCarry = rest
			newLine[i] = cell
			carry = nextCarry
		else:
			if carry >= 0:
				newLine[i] = carry
				carry = -1
	
	# return result
	return newLine

def iterate():
	# convert Golly format to internal format, with 12 empty cells to the left and right
	r = rect(g.getrect())
	y = r.top + r.height
	line = [-1] * (r.width + 24)
	for x in xrange(r.x, r.x + r.width):
		line[x - r.x + 12] = g.getcell(x, y - 1) - 1
		
	# apply rule
	newLine = calculateNextLine(line)
	
	# set next line
	for x in xrange(r.x - 12, r.x + r.width + 12):
		g.setcell(x, y, newLine[x - r.x + 12] + 1)

def undoStep():
	r = rect(g.getrect())
	y = r.top + r.height - 1

	for x in range(r.x, r.x + r.width):
		g.setcell(x, y, 0)


def gen():
	iterate()
	g.setgen("+1")

def step():
	i = g.getstep()
	if (i < 0):
		i = 1
	else:
		i = g.getbase() ** i
	for j in range(0, i):
		iterate()
	g.setgen("+" + str(i))

	return i

def undo():
	i = len(undoHistory)
	if i == 0: return
	i = undoHistory.pop(i - 1)
	for j in range(0, i):
		undoStep()
	g.setgen("-" + str(i))

def start():
	i = 0
	while True:
		if g.getkey() == '\r': break
		i += step()
		update(0)
	return i

# clear view and set algorithm and rule
g.new("SoupAutomat")   
g.setalgo("RuleTable")
g.setrule("SoupAutomat")

# get start line from user
line = g.getstring("Enter start line, e.g. 123...45 :")

# create start line
for i in range(len(line)):
	c = line[i]
	if c <> '.':
		g.setcell(i, 0, int(c) + 1)

if g.empty(): g.exit("There is no pattern.")

try:
	while True:
		ch = g.getkey()
		if ch == ' ':
			gen()
			update(1)
		elif ch == '\t':
			i = step()
			update(i)
		elif ch == 'z':
			undo()
			update(0)
		elif ch == '\r':
			i = start()
			update(i)
		elif len(ch) > 0:
			#g.show(str(len(ch)) +" "+str(ord(ch)))
			g.dokey(ch)
finally:
	pass
