#include <stdio.h>
#include <string.h>
#include "stm32f4_discovery.h"
#include "stm32f4xx_tim.h"
#include "hardware.h"

// I2S pins and ports
#define I2S_WS_PORT     GPIOA
#define I2S_WS_PIN      GPIO_Pin_4
#define I2S_WS_PINSRC   GPIO_PinSource4

#define I2S_CK_PORT     GPIOC
#define I2S_CK_PIN      GPIO_Pin_10
#define I2S_CK_PINSRC   GPIO_PinSource10

#define I2S_SDO_PORT    GPIOC
#define I2S_SDO_PIN     GPIO_Pin_12
#define I2S_SDO_PINSRC  GPIO_PinSource12
 
#define I2S_SDI_PORT    GPIOC
#define I2S_SDI_PIN     GPIO_Pin_11
#define I2S_SDI_PINSRC  GPIO_PinSource11

#define I2S_MCLK_PORT   GPIOC
#define I2S_MCLK_PIN    GPIO_Pin_7
#define I2S_MCLK_PINSRC GPIO_PinSource7


uint32_t __attribute__ ((aligned(16))) g_dacBuf[2][BUFFER_SIZE];
uint32_t __attribute__ ((aligned(16))) g_adcBuf[2][BUFFER_SIZE];

static void timerConfig()
{
	RCC_APB2PeriphClockCmd(RCC_APB2Periph_TIM1, ENABLE);
	RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM2, ENABLE);
	
	// TIM1 generate 1ms event
	TIM_TimeBaseInitTypeDef  TIM_TimeBaseStructure;
	TIM_TimeBaseStructInit(&TIM_TimeBaseStructure);
	TIM_TimeBaseStructure.TIM_Prescaler = 1;
	TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
	TIM_TimeBaseStructure.TIM_Period = SystemCoreClock / 1000000 / 2 - 1;
	TIM_TimeBaseInit(TIM1, &TIM_TimeBaseStructure);
	TIM_SelectOutputTrigger(TIM1, TIM_TRGOSource_Update);
	TIM_SelectMasterSlaveMode(TIM1, TIM_MasterSlaveMode_Enable);
	TIM_Cmd(TIM1, ENABLE);
	
	// TIM2 accept 1ms clock from TIM1
	TIM_TimeBaseStructInit(&TIM_TimeBaseStructure);
	TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
	TIM_TimeBaseStructure.TIM_Period = 0xffffffff;
	TIM_TimeBaseInit(TIM2, &TIM_TimeBaseStructure);
	TIM_SelectSlaveMode(TIM2, TIM_SlaveMode_External1);
	TIM_ITRxExternalClockConfig(TIM2, TIM_TS_ITR0);
	TIM_SelectOutputTrigger(TIM2, TIM_TRGOSource_Update);
	TIM_SelectMasterSlaveMode(TIM2, TIM_MasterSlaveMode_Enable);
	TIM_Cmd(TIM2, ENABLE);
}

uint32_t getTime()
{
	return TIM_GetCounter(TIM2);
}

void resetTime()
{
	TIM_SetCounter(TIM2, 0);
}

static void usartConfig()
{
	// enable clock for GPIO D and USART3
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD, ENABLE);
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOE, ENABLE);
	RCC_APB1PeriphClockCmd(RCC_APB1Periph_USART3, ENABLE);
	
	// configure and enable USART
	USART_InitTypeDef usart;
	USART_StructInit(&usart);
	usart.USART_BaudRate = 115200;
	usart.USART_WordLength = USART_WordLength_8b;
	usart.USART_StopBits = USART_StopBits_1;
	usart.USART_Parity = USART_Parity_No;
	usart.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
	usart.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
	USART_Init(USART3, &usart);
	USART_Cmd(USART3, ENABLE);
	
	// Configure USART3_Tx as 'alternate function' (AF)
	GPIO_InitTypeDef gpio;
	GPIO_StructInit(&gpio);
	gpio.GPIO_Pin = GPIO_Pin_8;
	gpio.GPIO_Mode = GPIO_Mode_AF;
	GPIO_Init(GPIOD, &gpio);
	
	// Configure USART3_Rx as 'input'
	gpio.GPIO_Pin = GPIO_Pin_9;
	GPIO_Init(GPIOD, &gpio);
	
	// Connect PD8 and PD9 pins to AF
	GPIO_PinAFConfig(GPIOD, GPIO_PinSource8, GPIO_AF_USART3);
	GPIO_PinAFConfig(GPIOD, GPIO_PinSource9, GPIO_AF_USART3);
	
	// turn off buffers, so IO occurs immediately
	setvbuf(stdin, NULL, _IONBF, 0);
	setvbuf(stdout, NULL, _IONBF, 0);
	setvbuf(stderr, NULL, _IONBF, 0);
}

void SysTick_Handler(void)
{
	tick();
}

static void i2sInit()
{
	// configure I2S output pins
	GPIO_InitTypeDef gpio;
	GPIO_StructInit(&gpio);
	gpio.GPIO_Speed = GPIO_Speed_50MHz;
	gpio.GPIO_Mode = GPIO_Mode_AF;
	gpio.GPIO_OType = GPIO_OType_PP;
	gpio.GPIO_Pin = I2S_WS_PIN;
	GPIO_Init(I2S_WS_PORT, &gpio);
	gpio.GPIO_Pin = I2S_CK_PIN;
	GPIO_Init(I2S_CK_PORT, &gpio);
	gpio.GPIO_Pin = I2S_SDO_PIN;
	GPIO_Init(I2S_SDO_PORT, &gpio);
	gpio.GPIO_Pin = I2S_MCLK_PIN;
	GPIO_Init(I2S_MCLK_PORT, &gpio);
	
	// configure I2S SDI input pin (IN doesn't work?)
	//gpio.GPIO_Mode = GPIO_Mode_IN;
	gpio.GPIO_Pin = I2S_SDI_PIN;
	GPIO_Init(I2S_SDI_PORT, &gpio);
	
	// set alternate functions
	GPIO_PinAFConfig(I2S_WS_PORT, I2S_WS_PINSRC, GPIO_AF_SPI3);
	GPIO_PinAFConfig(I2S_CK_PORT, I2S_CK_PINSRC, GPIO_AF_SPI3);
	GPIO_PinAFConfig(I2S_SDO_PORT, I2S_SDO_PINSRC, GPIO_AF_SPI3);
	GPIO_PinAFConfig(I2S_MCLK_PORT, I2S_MCLK_PINSRC, GPIO_AF_SPI3);
	GPIO_PinAFConfig(I2S_SDI_PORT, I2S_SDI_PINSRC, 5);
	
	// init I2S peripheral
	RCC_APB1PeriphClockCmd(RCC_APB1Periph_SPI3, ENABLE);
	RCC_PLLI2SCmd(ENABLE);
	I2S_InitTypeDef i2s;
	I2S_StructInit(&i2s);
	i2s.I2S_Standard = I2S_Standard_Phillips;
	i2s.I2S_DataFormat = I2S_DataFormat_32b;
	i2s.I2S_MCLKOutput = I2S_MCLKOutput_Enable;
	i2s.I2S_AudioFreq  = I2S_AudioFreq_96k;
	i2s.I2S_CPOL = I2S_CPOL_Low;
	i2s.I2S_Mode = I2S_Mode_MasterTx;
	I2S_Init(SPI3, &i2s);
	
	// init full duplex mode (I2S-ext for RX)
	I2S_FullDuplexConfig(I2S3ext, &i2s);

	// enable DMA for I2S3 and the ext receiver
    	SPI_I2S_DMACmd(SPI3, SPI_I2S_DMAReq_Tx, ENABLE);
	SPI_I2S_DMACmd(I2S3ext, SPI_I2S_DMAReq_Rx, ENABLE);

	// enable I2S3 sender and the ext receiver
	I2S_Cmd(SPI3, ENABLE);
	I2S_Cmd(I2S3ext, ENABLE);

	// init DMA for I2S Tx
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_DMA1, ENABLE);
	DMA_Cmd(DMA1_Stream5, DISABLE);
	DMA_ClearFlag(DMA1_Stream5, DMA_FLAG_TCIF5 | DMA_FLAG_TEIF5 | DMA_FLAG_HTIF5 | DMA_FLAG_FEIF5);
	DMA_InitTypeDef dma;
	DMA_StructInit(&dma);
	dma.DMA_Channel = DMA_Channel_0;
	dma.DMA_PeripheralBaseAddr = (u32)&SPI3->DR;
	dma.DMA_Memory0BaseAddr = (u32)g_dacBuf[0];
	dma.DMA_DIR = DMA_DIR_MemoryToPeripheral;
	dma.DMA_BufferSize = (uint32_t) 2 * BUFFER_SIZE;
	dma.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
	dma.DMA_MemoryInc = DMA_MemoryInc_Enable;
	dma.DMA_PeripheralDataSize = DMA_PeripheralDataSize_HalfWord;
	dma.DMA_MemoryDataSize = DMA_PeripheralDataSize_HalfWord;
	dma.DMA_Mode = DMA_Mode_Circular;
	dma.DMA_Priority = DMA_Priority_Medium;
	dma.DMA_FIFOMode = DMA_FIFOMode_Enable;
	dma.DMA_MemoryBurst = DMA_MemoryBurst_Single;
	dma.DMA_PeripheralBurst = DMA_PeripheralBurst_Single;
	DMA_Init(DMA1_Stream5, &dma);
	DMA_DoubleBufferModeConfig(DMA1_Stream5, (uint32_t)g_dacBuf[1], DMA_Memory_0);
	DMA_DoubleBufferModeCmd(DMA1_Stream5, ENABLE);

	// init DMA for I2S Rx
	DMA_Cmd(DMA1_Stream0, DISABLE);
	DMA_ClearFlag(DMA1_Stream0, DMA_FLAG_TCIF0 | DMA_FLAG_TEIF0 | DMA_FLAG_HTIF0 | DMA_FLAG_FEIF0);
	DMA_StructInit(&dma);
	dma.DMA_Channel = DMA_Channel_3;
	dma.DMA_PeripheralBaseAddr = (u32)&I2S3ext->DR;
	dma.DMA_Memory0BaseAddr = (u32)g_adcBuf[0];
	dma.DMA_DIR = DMA_DIR_PeripheralToMemory;
	dma.DMA_BufferSize = (uint32_t) 2 * BUFFER_SIZE;
	dma.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
	dma.DMA_MemoryInc = DMA_MemoryInc_Enable;
	dma.DMA_PeripheralDataSize = DMA_PeripheralDataSize_HalfWord;
	dma.DMA_MemoryDataSize = DMA_PeripheralDataSize_HalfWord;
	dma.DMA_Mode = DMA_Mode_Circular;
	dma.DMA_Priority = DMA_Priority_High;
	dma.DMA_FIFOMode = DMA_FIFOMode_Disable;
	dma.DMA_MemoryBurst = DMA_MemoryBurst_Single;
	dma.DMA_PeripheralBurst = DMA_PeripheralBurst_Single;
	DMA_Init(DMA1_Stream0, &dma);
	DMA_DoubleBufferModeConfig(DMA1_Stream0, (uint32_t)g_adcBuf[1], DMA_Memory_0);
	DMA_DoubleBufferModeCmd(DMA1_Stream0, ENABLE);

	// setup interrupt controller
	NVIC_InitTypeDef nvic;
	nvic.NVIC_IRQChannel = DMA1_Stream5_IRQn;
	nvic.NVIC_IRQChannelPreemptionPriority = 0;
	nvic.NVIC_IRQChannelSubPriority = 0;
	nvic.NVIC_IRQChannelCmd = ENABLE;
	NVIC_Init(&nvic);

	// enable interrupt on half transfer and transfer complete
	DMA_ClearITPendingBit(DMA1_Stream5, DMA_FLAG_TCIF5);
	DMA_ITConfig(DMA1_Stream5, DMA_IT_TC, ENABLE);
	
	// setup interrupt controller for Rx
	nvic.NVIC_IRQChannel = DMA1_Stream0_IRQn;
	NVIC_Init(&nvic);

	// enable interrupt on transfer complete for Rx
	DMA_ClearITPendingBit(DMA1_Stream0, DMA_FLAG_TCIF0);
	DMA_ITConfig(DMA1_Stream0, DMA_IT_TC, ENABLE);
	
	// start DMA transfer
	DMA_Cmd(DMA1_Stream5, ENABLE);
	DMA_Cmd(DMA1_Stream0, ENABLE);
}
	
static void swapWords(uint32_t* samples, uint32_t count)
{
	int i;
	for (i = 0; i < count; i++) {
		uint32_t sample = samples[i];
		samples[i] = (sample >> 16) | ((sample & 0xffff) << 16);
	}
}

void DMA1_Stream5_IRQHandler(void)
{
	GPIO_SetBits(GPIOE, GPIO_Pin_0);
	
	// clear interrupts
	if (DMA_GetFlagStatus(DMA1_Stream5, DMA_IT_TCIF5) == SET) DMA_ClearITPendingBit(DMA1_Stream5, DMA_FLAG_TCIF5);
	
	// check which buffer can be used
	int i = DMA_GetCurrentMemoryTarget(DMA1_Stream5) == 0 ? 1 : 0;
	
	// fill it
	onFillSamples(g_dacBuf[i], BUFFER_SIZE);
	swapWords(g_dacBuf[i], BUFFER_SIZE);
	
	GPIO_ResetBits(GPIOE, GPIO_Pin_0);
}

void DMA1_Stream0_IRQHandler(void)
{
//	GPIO_SetBits(GPIOE, GPIO_Pin_0);
	
	// clear interrupts
	if (DMA_GetFlagStatus(DMA1_Stream0, DMA_IT_TCIF0) == SET) DMA_ClearITPendingBit(DMA1_Stream0, DMA_FLAG_TCIF0);
	
	// check which buffer can be used
	int i = DMA_GetCurrentMemoryTarget(DMA1_Stream0) == 0 ? 1 : 0;
	
	// process it, but ignore the memory garbage on startup
	static int startupCounter = 0;
	if (startupCounter < 2) {
		startupCounter++;
	} else {
		swapWords(g_adcBuf[i], BUFFER_SIZE);
		onSamplesReceived(g_adcBuf[i], BUFFER_SIZE);
	}
	
//	GPIO_ResetBits(GPIOE, GPIO_Pin_0);
}

void initHardware()
{
	// required for VFP
	SCB->CPACR |= ((3UL << 10*2)|(3UL << 11*2));
	
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOA, ENABLE);
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOB, ENABLE);
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOC, ENABLE);
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD, ENABLE);
	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOE, ENABLE);
	
	GPIO_InitTypeDef gpio;
	gpio.GPIO_Pin = GPIO_Pin_12 | GPIO_Pin_13;
	gpio.GPIO_Mode = GPIO_Mode_OUT;
	gpio.GPIO_OType = GPIO_OType_PP;
	gpio.GPIO_Speed = GPIO_Speed_2MHz;
	gpio.GPIO_PuPd = GPIO_PuPd_NOPULL;
	GPIO_Init(GPIOD, &gpio);
	
	gpio.GPIO_Pin = GPIO_Pin_0;
	GPIO_Init(GPIOE, &gpio);
	
	gpio.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_1;
	GPIO_Init(GPIOD, &gpio);
	
	timerConfig();
	
	usartConfig();

	// fill first dac buffer		
	onFillSamples(g_dacBuf[0], BUFFER_SIZE);
	swapWords(g_dacBuf[0], BUFFER_SIZE);
	
	i2sInit();
	
	// enable 1 ms SysTick timer	
	SysTick_CLKSourceConfig(SysTick_CLKSource_HCLK);
	SysTick_Config(SystemCoreClock / 1000);
}

void setGreenLed(int state)
{
	if (state) {
		GPIO_SetBits(GPIOD, GPIO_Pin_12);
	} else {
		GPIO_ResetBits(GPIOD, GPIO_Pin_12);
	}
}

int isUserButtonPressed()
{
	return GPIO_ReadInputDataBit(GPIOA, GPIO_Pin_0) == Bit_SET;
}

void setPortD(uint32_t bits)
{
	GPIO_Write(GPIOD, bits);
}

int uartIsCharAvailable()
{
	return ((USART3->SR & USART_FLAG_RXNE) != (uint16_t)RESET);
}

uint8_t uartGetChar()
{
	return (char)(USART3->DR & (uint16_t)0x01FF);
}
